--------------------------------------------------------------------------------------------------------------
-- Gile[s] importer for 3DS Max
--
-- Author:     Mikkel Fredborg ( giles@frecle.net, http://www.frecle.net )
-- Extended by A. Botorabi     ( botorabi@gmx.net, http://yag2002.sourceforge.net  )
--
-- Date of Creation   1. November 2004
-- Last modification  9. November 2004
--------------------------------------------------------------------------------------------------------------

-- ChangeLog
--------------------------------------------------------------------------------------------------------------

-- 9. Nov 2004    botorabi     + added support for choosing the lightmap channel on import ( 2..99 )
--                             + better lightmap image name generation: [gls file name]"_lm"[auto index]".png"

--------------------------------------------------------------------------------------------------------------

global GLS_HEADER					= 0xFFFF
global GLS_AUTHOR					= 0xF000
global GLS_EXPORTFILE				= 0xF001

global GLS_NODES					= 0x1000
global GLS_NODE						= 0x1001
global GLS_NODE_NAME				= 0x1002
global GLS_NODE_POSITION			= 0x1003
global GLS_NODE_ROTATION			= 0x1004
global GLS_NODE_SCALE				= 0x1005
global GLS_NODE_CUSTOMPROPS			= 0x1006
global GLS_NODE_FILE				= 0x1007
global GLS_NODE_HIDDEN				= 0x1008

global GLS_MESH						= 0x2000
global GLS_MESH_OVERRIDE			= 0x2001
global GLS_MESH_RECEIVEBACK			= 0x2002
global GLS_MESH_RECEIVESHADOW		= 0x2003
global GLS_MESH_CASTSHADOW			= 0x2004
global GLS_MESH_RECEIVEGLOBAL		= 0x2005
global GLS_MESH_AFFECTGLOBAL		= 0x2006

global GLS_MESH_SURFACES			= 0x2100
global GLS_MESH_SURF				= 0x2101
global GLS_MESH_SURFVERTS			= 0x2102
global GLS_MESH_SURFPOLYS			= 0x2103
global GLS_MESH_SURFMATERIAL		= 0x2104
global GLS_MESH_SURFVERTFORMAT		= 0x2105
global GLS_MESH_SURFVERTDATA		= 0x2106
global GLS_MESH_SURFPOLYDATA		= 0x2107

global GLS_PIVOT					= 0x3000

global GLS_LIGHT					= 0x4000
global GLS_LIGHT_TYPE				= 0x4001
global GLS_LIGHT_ACTIVE				= 0x4002
global GLS_LIGHT_CASTSHADOWS		= 0x4003
global GLS_LIGHT_INFINITE			= 0x4004
global GLS_LIGHT_OVERSHOOT			= 0x4005
global GLS_LIGHT_RADIUS				= 0x4006
global GLS_LIGHT_RED				= 0x4007
global GLS_LIGHT_GREEN				= 0x4008
global GLS_LIGHT_BLUE				= 0x4009
global GLS_LIGHT_INTENSITY			= 0x400A
global GLS_LIGHT_NEAR				= 0x400B
global GLS_LIGHT_FAR				= 0x400C
global GLS_LIGHT_INNER				= 0x400D
global GLS_LIGHT_OUTER				= 0x400E
global GLS_LIGHT_TOON				= 0x400F
global GLS_LIGHT_TOONLEVELS			= 0x4010
global GLS_LIGHT_SHOWFALLOFF		= 0x4011

global GLS_MATERIALS				= 0x5000
global GLS_MAT						= 0x5001
global GLS_MAT_NAME					= 0x5002
global GLS_MAT_RED					= 0x5003
global GLS_MAT_GREEN				= 0x5004
global GLS_MAT_BLUE					= 0x5005
global GLS_MAT_ALPHA				= 0x5006
global GLS_MAT_SELFILLUMINATION		= 0x5007
global GLS_MAT_SHININESS			= 0x5008
global GLS_MAT_FX					= 0x5009
global GLS_MAT_BLEND				= 0x500A
global GLS_MAT_LIGHTMETHOD			= 0x500B
global GLS_MAT_LIGHTMAP				= 0x500C
global GLS_MAT_RECEIVEBACK			= 0x500D
global GLS_MAT_RECEIVESHADOW		= 0x500E
global GLS_MAT_CASTSHADOW			= 0x500F
global GLS_MAT_RECEIVEGLOBAL		= 0x5010
global GLS_MAT_AFFECTGLOBAL			= 0x5011
global GLS_MAT_TEXLAYER				= 0x5012
global GLS_MAT_GEOENABLE			= 0x5013
global GLS_MAT_GEORANGE				= 0x5014

global GLS_TEXTURES					= 0x6000
global GLS_TEX						= 0x6001
global GLS_TEX_FILE					= 0x6002
global GLS_TEX_SCALEU				= 0x6003
global GLS_TEX_SCALEV				= 0x6004
global GLS_TEX_OFFSETU				= 0x6005
global GLS_TEX_OFFSETV				= 0x6006
global GLS_TEX_ANGLE				= 0x6007
global GLS_TEX_FLAGS				= 0x6008
global GLS_TEX_BLEND				= 0x6009
global GLS_TEX_COORDSET				= 0x600A

global GLS_LIGHTMAPS				= 0x7000
global GLS_LMAP						= 0x7001
global GLS_LMAP_NAME				= 0x7002
global GLS_LMAP_FILE				= 0x7003
global GLS_LMAP_WIDTH				= 0x7004
global GLS_LMAP_HEIGHT				= 0x7005
global GLS_LMAP_NONUNIFORM			= 0x7006
global GLS_LMAP_USECUSTOMTEXEL		= 0x7007
global GLS_LMAP_CUSTOMTEXEL			= 0x7008
global GLS_LMAP_REPACK				= 0x7009
global GLS_LMAP_DATA				= 0x700A
global GLS_LMAP_PACKMETHOD			= 0x700B
global GLS_LMAP_FLATMETHOD			= 0x700C
global GLS_LMAP_FLATAXIS			= 0x700D

global GLS_RENDER					= 0x8000
global GLS_RENDER_CLEARBEFORE		= 0x8001
global GLS_RENDER_DIRECTENABLE		= 0x8002
global GLS_RENDER_GLOBALENABLE		= 0x8003
global GLS_RENDER_RAYBIAS			= 0x8004
global GLS_RENDER_DIRECTMULTIPLY	= 0x8005
global GLS_RENDER_DIRECTBACKSHADOW	= 0x8006
global GLS_RENDER_DIRECTSHADOWS		= 0x8007
global GLS_RENDER_DIRECTSOFT		= 0x8008
global GLS_RENDER_DIRECTSOFTSAMPLES	= 0x8009
global GLS_RENDER_GLOBALIGNORETEX	= 0x800A
global GLS_RENDER_GLOBALITERATIONS	= 0x800B
global GLS_RENDER_GLOBALDENSITY		= 0x800C
global GLS_RENDER_GLOBALSAMPLES		= 0x800D
global GLS_RENDER_GLOBALMULTIPLY	= 0x800E
global GLS_RENDER_SKYENABLE			= 0x800F
global GLS_RENDER_SKYRED			= 0x8010
global GLS_RENDER_SKYGREEN			= 0x8011
global GLS_RENDER_SKYBLUE			= 0x8012
global GLS_RENDER_SKYMULTIPLY		= 0x8013
global GLS_RENDER_AUTOBLUR			= 0x8014
global GLS_RENDER_AUTOEXPAND		= 0x8015
global GLS_RENDER_AUTOBLURRADIUS	= 0x8016
global GLS_RENDER_SKYUSETEXTURE		= 0x8017
global GLS_RENDER_SKYTEXTURE		= 0x8018
global GLS_RENDER_LEAKPREVENTION	= 0x8019
global GLS_RENDER_GEOENABLE			= 0x801A
global GLS_RENDER_GEOSAMPLES		= 0x801B
global GLS_RENDER_ANTIALIAS			= 0x801C

-- uv channel for lm layer
global GLS_LM_MAP_CHANNEL      = 3
	
global gls_matloader = 2
	
global gls_ChunkStack 	= #()
global gls_TOS 			= 1
global gls_ChunkID		= 0
global gls_ChunkSize	= 0
global gls_FileSize		= 0
	
global gls_texture  = #()
global gls_alphatex = #()
global gls_lightmap = #()
global gls_material = #()
global gls_multimat


utility giles_importer "gile[s] scene importer" width:160 height:240
(
	button doimp "Import gile[s] scene" pos:[5,235] width:150 height:20
	label lbl2 "gls importer version 1.0" pos:[22,59] width:120 height:15
	GroupBox grp1 "Lightmap Importing" pos:[5,149] width:150 height:75
	label lbl3 "http://www.frecle.net/giles" pos:[15,73] width:129 height:15
	radiobuttons rdo2 "" pos:[10,166] width:126 height:32 labels:#("Self Illumination", "Diffuse->RGB Multiply") default:2	
    spinner lm_channel_spinner "" pos:[10,200] width:32 height:16 range:[2,99,3] type:#integer scale:1
    label spinner_caption "Lightmap Channel" pos:[45,200] width:90 height:16	
	bitmap bmp1 "Bitmap" pos:[5,5] width:150 height:51 fileName:"gile[s]_logo.png"
	GroupBox grp2 "Scene Importing" pos:[5,92] width:150 height:55
	radiobuttons rdo3 "" pos:[11,108] width:109 height:32 labels:#("Reset Scene", "Merge with Scene") default:2
	
	-- callback for lm channel spinner
	on lm_channel_spinner changed new_value do 
	( 
		GLS_LM_MAP_CHANNEL = new_value
	)
	
	-------------------------
	-- Import functions
	-------------------------
	function gls_ReadChunk filestream = (
	
		if getProgressCancel() do
		(
			gls_ChunkID = 0
			fseek filestream 0 #seek_end
			return undefined
		)

		-- Check if we have left any chunks
		local remove = 0
		If (gls_TOS>1) then
		( 
			For i = 2 to gls_tos do
			(
				If (ftell filestream >= gls_ChunkStack[i]) then
				(
					remove = remove + 1
				)
			)
			gls_tos = gls_tos - remove
		)
		
		gls_ChunkID		= ReadLong filestream
		gls_ChunkSize	= ReadLong filestream
		gls_TOS			= gls_TOS+1
		local thispos = ftell filestream
		gls_ChunkStack[gls_TOS] = thispos+gls_ChunkSize
	
		val = ftell filestream
		progressUpdate (val*100 / gls_FileSize)
		
		--format "ChunkID:%, ChunkSide:% \n" gls_chunkid gls_chunksize

		if (ftell filestream < gls_FileSize) then
		(	
			Return 1
		)else(
			Return 0
		)
	)
	
	function gls_ExitChunk filestream = (

		if getProgressCancel() do
		(
			fseek filestream 0 #seek_end
			return undefined
		)
								
		s = bit.intAsHex gls_ChunkID
		--format "gls_ExitChunk on ChunkID: % \n " s
		fseek filestream gls_ChunkStack[gls_TOS] #seek_set
		gls_TOS = gls_TOS-1
		
	)
	
	function gls_ReadLightmaps filestream = (
	
		local motherchunk = gls_tos

	 -- variables for auto naming lm file name
	 local lm_filename   = "lm" as string
       local lm_file_index = 0
		
		While (ftell filestream < gls_ChunkStack[motherchunk]) do
		(
			gls_ReadChunk filestream
			
			case gls_chunkid of
			(
				-- New lightmap
				GLS_LMAP:
					bob = 1
					
				-- Lightmap Name
				GLS_LMAP_NAME:
					nam = ReadString filestream
			
				-- Lightmap Export File
				GLS_LMAP_FILE:
					(
						fil = ReadString filestream
						if (fil.count < 4) do
						(						    
							-- auto generate lightmap file name: [gls file name]"_lm"[auto incremented index]".png"
							fil = nam + "_" + lm_filename + lm_file_index as string + ".png"
							lm_file_index = lm_file_index + 1
							format "importing lightmap file: % \n" fil
							print $outp 
						)
						
					)
									
				-- Lightmap Width
				GLS_LMAP_WIDTH:
					w = ReadShort filestream
		
				-- Lightmap Height
				GLS_LMAP_HEIGHT:
					h = ReadShort filestream
					
				-- Lightmap Repack
				GLS_LMAP_REPACK:
					repack = ReadByte filestream
					
				-- Lightmap non-uniform flag
				GLS_LMAP_NONUNIFORM:				
					nu = ReadByte filestream
				
				-- Lightmap custom texel flag	
				GLS_LMAP_USECUSTOMTEXEL:
					uc = ReadByte filestream
					
				-- Lightmap custom texel size
				GLS_LMAP_CUSTOMTEXEL:
					ct = ReadFloat filestream
						
				-- Lightmap packing method
				GLS_LMAP_PACKMETHOD:
					flat = ReadByte filestream
	
				-- Lightmap flat packing method
				GLS_LMAP_FLATMETHOD:
					flatbest = ReadByte filestream
	
				-- Lightmap flat packing method
				GLS_LMAP_FLATAXIS:
					flataxis = ReadByte filestream
							
				-- Lightmap colors (width*height*3 bytes)
				GLS_LMAP_DATA:
					(
						bmap = bitmap w h filename:fil
						For x = 0 To (w-1) do
						(
							For y = 0 To (h-1) do
							(	
								r = ReadByte filestream			
								g = ReadByte filestream
								b = ReadByte filestream
								
								SetPixels bmap [x,y] #([r,g,b] as color)
								if getProgressCancel() do
								(
									fseek filestream 0 #seek_end
									return undefined
								)
							)
							val = ftell filestream
							if ((mod x 16)==0) do
							(
								progressUpdate (val*100 / gls_FileSize)
							)	
							--format "filestream:%\n" filestream
						)
						save bmap
						lmap = bitmaptexture()
						append gls_lightmap lmap
						lmap_id = gls_lightmap.count
						gls_lightmap[lmap_id].bitmap = bmap
						gls_lightmap[lmap_id].output.RGB_Offset = 0.0
						gls_lightmap[lmap_id].output.RGB_Level	= 2.0
						gls_lightmap[lmap_id].coords.mapChannel = GLS_LM_MAP_CHANNEL
						gls_lightmap[lmap_id].filtering 		= 0
						gls_lightmap[lmap_id].name				= nam
					)
		
				Default:
					gls_ExitChunk filestream
					
			)
		)
	)
	
	function gls_ReadTextures filestream = (
	
		local motherchunk = gls_tos
		While (ftell filestream < gls_ChunkStack[motherchunk]) do
		(
			gls_ReadChunk filestream
			
			case gls_chunkid of
			(
				-- New texture
				GLS_TEX:
					(
						tex = bitmaptexture()
						append gls_texture tex
						append gls_alphatex undefined
						
						tex_id = gls_texture.count
						
						gls_texture[tex_id].filtering = 0
						
						tname = tex_id as string
						gls_texture[tex_id].name		= "Texture #"+tname
						gls_texture[tex_id].alphasource	= 2
						--format "TextureID: %\n" tex_id
						--format "TextureName: %\n" gls_texture[tex_id].name
					)
					
				-- Texture file
				GLS_TEX_FILE:
					(
						gls_texture[tex_id].filename = readString filestream
		--				gls_texture[tex_id].reload()
					)
	
				-- Texture Scale U
				GLS_TEX_SCALEU: 
					su = ReadFloat filestream
					
				-- Texture Scale V
				GLS_TEX_SCALEV: 
					sv = ReadFloat filestream
					
				-- Texture Offset U
				GLS_TEX_OFFSETU: 
					ou = ReadFloat filestream
					
				-- Texture Offset V
				GLS_TEX_OFFSETV: 
					ov = ReadFloat filestream
					
				-- Texture Angle W
				GLS_TEX_ANGLE: 
					(
						ang = ReadFloat filestream
	
						ux = Sin(ang)/su/2.0
						uy = Cos(ang)/su/2.0
						vx = Cos(ang)/sv/2.0
						vy = Sin(ang)/sv/2.0
						
						c0 = (ou + 0.5 - ux + uy)
						c1 = (ov + 0.5 + vx - vy)
						
						a0 = Cos(ang)/su 
						b0 = Sin(ang)/su 
						
						a1 = Sin(ang)/sv 
						b1 = Cos(ang)/sv 
						
						ou = (c0*b1 - b0*c1) / (a0*b1 - b0*a1)
						ov = (a0*c1 - c0*a1) / (a0*b1 - b0*a1)
	
						gls_texture[tex_id].coordinates.U_Offset 	= ou
						gls_texture[tex_id].coordinates.V_Offset 	= ov
						gls_texture[tex_id].coordinates.U_Tiling 	= 1.0/su
						gls_texture[tex_id].coordinates.V_Tiling 	= 1.0/sv
						gls_texture[tex_id].coordinates.W_Angle		= ang
					)
					
				-- Texture Flags
				GLS_TEX_FLAGS: 
					(
						flags = ReadLong filestream
						
						-- Alpha
						if ((bit.and flags 2)!=0) do
						(
							gls_alphatex[tex_id] = copy gls_texture[tex_id]
							gls_alphatex[tex_id].name = "AlphaTex #"+tname
							gls_alphatex[tex_id].alphasource = 0
							gls_alphatex[tex_id].monoOutput = 1
							gls_alphatex[tex_id].RGBOutput	= 1
						)
						-- Masked
						if ((bit.and flags 4)!=0) do
						(
							gls_alphatex[tex_id] = copy gls_texture[tex_id]
							gls_alphatex[tex_id].name = "MaskedTex #"+tname
							gls_alphatex[tex_id].alphasource 		= 2
							gls_alphatex[tex_id].monoOutput 		= 0
							gls_alphatex[tex_id].RGBOutput			= 0
							gls_alphatex[tex_id].filtering			= 2
							gls_alphatex[tex_id].output.RGB_Level 	= 255.0
							gls_alphatex[tex_id].output.clamp 	 	= true
						)
						-- Spherical Reflection map
						if ((bit.and flags 64)!=0) do
						(
							gls_texture[tex_id].coordinates.mappingType = 1
							gls_texture[tex_id].coordinates.mapping = 0
						)
						-- Cubic Reflection map
						if ((bit.and flags 128)!=0) do
						(
							gls_texture[tex_id].coordinates.mappingType = 1
							gls_texture[tex_id].coordinates.mapping = 0
						)
					)
					
				-- Texture Blend
				GLS_TEX_BLEND: 
					ReadLong filestream
					
				-- Texture Coordinate Set
				GLS_TEX_COORDSET:
					ReadByte filestream
				
				default:
					gls_ExitChunk(True)
			)
		)
	)
	
	function gls_ReadMaterials filestream =(
	
		local mat_id = 0
	
		local motherchunk = gls_tos
		While (ftell filestream < gls_ChunkStack[motherchunk]) do
		(
			gls_ReadChunk filestream
			
			case gls_chunkid of
			(
				-- New material entry
				GLS_MAT:
					(
						mat = standardMaterial()
						append gls_material mat
						
						mat_id = gls_material.count
						
						mname = mat_id as string
						gls_material[mat_id].name = "Material #"+mname
						
						gls_material[mat_id].shaderType = 1
						gls_material[mat_id].specular = color 255 255 255
						gls_material[mat_id].ambient = color 255 255 255
						gls_material[mat_id].showInViewport = true
						
						setMTLMEditFlags gls_material[mat_id] #{2,3}
						if (mat_id<24) do
						(
							setMeditMaterial (mat_id+1) gls_material[mat_id]
						)
						
						if (gls_matloader==2) do
						(
							multmap = rgbMult()
							gls_material[mat_id].diffuseMap = multmap
							multmap.alphaFrom = 1
							multmap.alphaFrom = 0
						)
					)
					
				-- Material Name
				GLS_MAT_NAME:
					gls_material[mat_id].name = ReadString filestream
						
				-- Material Red
				GLS_MAT_RED:
					r = ReadFloat filestream
				
				-- Material Green
				GLS_MAT_GREEN:
					g = ReadFloat filestream
							
				-- Material Blue
				GLS_MAT_BLUE:
					(
						b = ReadFloat filestream
						r = r*255
						g = g*255
						b = b*255
						gls_material[mat_id].diffuse = color r g b
						if (gls_matloader==2) do
						(
							gls_material[mat_id].diffuseMap.color1 = color r g b
						)
					)
					
				-- Material Alpha
				GLS_MAT_ALPHA:
					gls_material[mat_id].opacity = (ReadFloat filestream)*100
							
				-- Material Self Illumination
				GLS_MAT_SELFILLUMINATION:
					gls_material[mat_id].selfIllumAmount = (ReadFloat filestream)*100
			
				-- Material Shininess
				GLS_MAT_SHININESS:
					(
						shininess = ReadFloat filestream
						gls_material[mat_id].specularLevel = shininess*100
					)
					
				-- Material FX
				GLS_MAT_FX:
					(
						fx = ReadLong filestream
						if ((bit.and fx 1)!=0) do
						(
							gls_material[mat_id].selfIllumAmount = 100
						)
						if ((bit.and fx 4)!=0) do
						(
							gls_material[mat_id].faceted = true
						)
						if ((bit.and fx 16)!=0) do
						(
							gls_material[mat_id].twoSided = true
						)
					)
							
				-- Material Blend
				GLS_MAT_BLEND:
					(
						blend = ReadLong filestream
						case blend of
						(
							1: gls_material[mat_id].opacityType = 0
							2: gls_material[mat_id].opacityType = 1
							3: gls_material[mat_id].opacityType = 2
						)
					)
							
				-- Light method (None = 0 / Vertex = 1 / Lightmap = 2)
				GLS_MAT_LIGHTMETHOD:
					(
						lightmethod = ReadByte filestream
						if (lightmethod == 1) do
						(
							vcol = vertex_color()
							if (gls_matloader==2) then
							(
								multmap.map2 = vcol
							)else(
								gls_material[mat_id].selfIllumMap = vcol
							)
						)
					)
	
				-- Receive Back Light
				GLS_MAT_RECEIVEBACK:
					receiveback = ReadByte filestream
	
				-- Receive Shadows
				GLS_MAT_RECEIVESHADOW:
					receiveshadows = ReadByte filestream
							
				-- Cast Shadows
				GLS_MAT_CASTSHADOW:
					castshadows = ReadByte filestream
							
				-- Receive Global Illumination
				GLS_MAT_RECEIVEGLOBAL:
					receivegi = ReadByte filestream
							
				-- Affect Global Illumination
				GLS_MAT_AFFECTGLOBAL:
					affectgi = ReadByte filestream
	
				-- Enable Geometry Light Boosting
				GLS_MAT_GEOENABLE:
					geoenable = ReadByte filestream
	
				-- Geometry Light Boosting Range
				GLS_MAT_GEORANGE:
					georange = ReadFloat filestream
											
				-- Material Lightmap ID
				GLS_MAT_LIGHTMAP:
					(
						lmap_id = ReadShort filestream
						--format "Lightmap ID on material: %\n" lmap_id
						if (gls_matloader==2) then
						(
							multmap.map2 = gls_lightmap[lmap_id]
							showTextureMap gls_material[mat_id] gls_material[mat_id].diffuseMap.map2 true
						)else(
							gls_material[mat_id].selfIllumMap = gls_lightmap[lmap_id]
							showTextureMap gls_material[mat_id] gls_material[mat_id].selfIllumMap true
						)
					)
				
				-- Material Texture IDs
				GLS_MAT_TEXLAYER:
					(
						tex_l  = ReadByte filestream
						tex_id = ReadShort filestream
						tex_id = tex_id + 1
						if (tex_l == 0) do
						(
							if (gls_texture[tex_id].coordinates.mappingType == 0) then
							(
								-- Standard diffuse map
								if (gls_matloader==2) then
								(
									multmap.map1 = gls_texture[tex_id]
									showTextureMap gls_material[mat_id] gls_material[mat_id].diffuseMap.map1 true
								)else(
									gls_material[mat_id].diffuseMap = gls_texture[tex_id]
									showTextureMap gls_material[mat_id] gls_material[mat_id].diffuseMap true
								)
							)else(
								-- Reflection map
								gls_material[mat_id].reflectionMap = gls_texture[tex_id] 
							)
							
							if (gls_alphatex[tex_id] != undefined) do
							(
								gls_material[mat_id].opacityMap = gls_alphatex[tex_id] 
							)
						)
					)
	
				Default:
					gls_ExitChunk filestream
			)
		)
		
		gls_multimat = multimaterial()
		gls_multimat.numsubs = gls_material.count
		for i = 1 to gls_material.count do
		(
				gls_multimat.materialList[i] = gls_material[i]
		)
		setMeditMaterial 1 gls_multimat
	)
	
	function gls_ReadSurfaces filestream =(
	
		local vertpos = #()
		local vertnor = #()
		local vertcol = #()
		local vertuv0 = #()
		local vertuv1 = #()
		
		local tris	= #()
		local trimat = #()
		
		local trivertadd = 1
	
		--format "Array Size: %\n" (tris.count)
	
		local motherchunk = gls_tos
		While (ftell filestream < gls_ChunkStack[motherchunk]) do
		(
			gls_ReadChunk filestream
			
			case gls_chunkid of
			(
				GLS_MESH_SURF:
					(
						brushid = 0
					)
			
				-- Number of surface vertices
				GLS_MESH_SURFVERTS:
					(
						n_verts = ReadShort filestream
					)
					
				-- Number of surface triangles
				GLS_MESH_SURFPOLYS:
					(
						n_tris = ReadShort filestream
					)
					
				-- Surface material
				GLS_MESH_SURFMATERIAL:
					(
						brushid = ReadLong filestream
						brushid = brushid+1
						--format "Material ID: %\n" brushid
					)
			
				-- Number of surface vertices
				GLS_MESH_SURFVERTFORMAT:
					vertformat = ReadLong filestream
					
				GLS_MESH_SURFVERTDATA:
					(
						For v = 1 To n_verts do
						(	
							-- Position 
							x = ReadFloat filestream
							y = ReadFloat filestream
							z = ReadFloat filestream		
							
							append vertpos [x,z,y]
							
							-- Normal
							If (bit.and vertformat 1)!= 0 then
							(
								nx = ReadFloat filestream
								ny = ReadFloat filestream
								nz = ReadFloat filestream
							)else(
								nx = 0.0
								ny = 1.0
								nz = 0.0
							)
	
							append vertnor [nx,nz,ny]
									
							-- Color + Alpha
							If (bit.and vertformat 2)!= 0 then
							(
								r = ReadByte filestream #unsigned
								g = ReadByte filestream #unsigned
								b = ReadByte filestream #unsigned
								a = ReadByte filestream #unsigned
							)else(
								r = 255
								g = 255
								b = 255
								a = 255
							)
							
							-- Vertex Light Data
							If (bit.and vertformat 4)!= 0 then
							(
								lr = ReadByte filestream #unsigned
								lg = ReadByte filestream #unsigned
								lb = ReadByte filestream #unsigned
							)else(
								lr = 255
								lg = 255
								lb = 255
							)
	
							r = r * lr/127.5
							g = g * lg/127.5
							b = b * lb/127.5
							if r>255 do r=255
							if g>255 do g=255
							if b>255 do b=255
							a = a
							
							append vertcol [r/255.0,g/255.0,b/255.0] 
							
							-- Lightmap Coordinates
							If (bit.and vertformat 8)!= 0 then
							(					
								u1 	= ReadFloat filestream
								v1	= ReadFloat filestream
							)else(
								u1	= 0.0
								v1	= 0.0
							)
							append vertuv1 [u1,-v1,1]
							
							-- Texture Coordinates 0
							If (bit.and vertformat 16)!= 0 then
							(
								u0 	= ReadFloat filestream
								v0	= ReadFloat filestream
							)else(
								u0	= 0.0
								v0	= 0.0
							)
							append vertuv0 [u0,-v0,1]
						)
					)
			
				GLS_MESH_SURFPOLYDATA:
					(
						For t = 1 To n_tris do
						(
							vert0 		= ReadShort filestream #unsigned
							vert1 		= ReadShort filestream #unsigned
							vert2 		= ReadShort filestream #unsigned		
	
							append tris [vert0+trivertadd,vert2+trivertadd,vert1+trivertadd]
							append trimat brushid
						)
						trivertadd = trivertadd + n_verts
					)
					
				Default:
					gls_ExitChunk filestream
			)
		)
	
		n_verts = vertpos.count
		n_tris	= tris.count
		
		-- Create mesh
		mm = mesh vertices:vertpos faces:tris materialIDs:trimat
	
		setCVertMode mm true
	
		-- Set to use Multi Material
		mm.material	= gls_multimat
	
		-- Set UV coordinates
		meshop.setNumMaps mm ( GLS_LM_MAP_CHANNEL + 1 )
	
		meshop.setMapSupport  mm 0 true
		meshop.setNumMapVerts mm 0 n_verts
		meshop.setNumMapFaces mm 0 n_tris
		
		meshop.setMapSupport  mm 1 true
		meshop.setNumMapVerts mm 1 n_verts
		meshop.setNumMapFaces mm 1 n_tris
	
		meshop.setMapSupport  mm GLS_LM_MAP_CHANNEL true
		meshop.setNumMapVerts mm GLS_LM_MAP_CHANNEL n_verts
		meshop.setNumMapFaces mm GLS_LM_MAP_CHANNEL n_tris	
	
		for i = 1 to n_verts do 
		(
			meshop.setMapVert mm 0 i vertcol[i]
			meshop.setMapVert mm 1 i vertuv0[i]
			meshop.setMapVert mm GLS_LM_MAP_CHANNEL i vertuv1[i]
		)
		for i = 1 to n_tris do 
		(
			meshop.setMapFace mm 0 i tris[i]
			meshop.setMapFace mm 1 i tris[i]
			meshop.setMapFace mm GLS_LM_MAP_CHANNEL i tris[i]
		)
		update mm
		
		return mm
	)
	
	function gls_ReadMesh filestream =(
	
		local motherchunk = gls_tos
		While (ftell filestream < gls_ChunkStack[motherchunk]) do
		(
			gls_ReadChunk filestream
			
			case gls_chunkid of
			(
				GLS_MESH_SURFACES:
					model = gls_ReadSurfaces filestream
					
				Default:
					gls_ExitChunk filestream
			)
		)
		
		return model
	)
	
	function gls_ReadPivot filestream =(
	
		local motherchunk = gls_tos
		While (ftell filestream < gls_ChunkStack[motherchunk]) do
		(
			gls_ReadChunk filestream
			
			case gls_chunkid of
			(
				Default:
					gls_ExitChunk filestream
			)
		)
		
		model = dummy()
		
		return model
	)
	
	function gls_ReadLight filestream =(
	
		local motherchunk = gls_tos
		While (ftell filestream < gls_ChunkStack[motherchunk]) do
		(
			gls_ReadChunk filestream
			
			case gls_chunkid of
			(	
				-- Light Class
				GLS_LIGHT_TYPE:
					(
						b = ReadByte filestream
						case b of
						(
							1: model = directionalLight()
							2: model = omniLight()
							3: model = freeSpot()
							4: model = omniLight()
						)
					)
			 		
				-- Active
				GLS_LIGHT_ACTIVE:
					(
						b = ReadByte filestream
						if (b==0) then
						(
							model.enabled = false
						)else(
							model.enabled = true
						)
					)
					
				-- Cast Shadows
				GLS_LIGHT_CASTSHADOWS:
					(
						b = ReadByte filestream
						if (b==0) then
						(
							model.baseObject.castshadows = false
						)else(
							model.baseObject.castshadows = true
						)
					)
			 
				-- Infinite
				GLS_LIGHT_INFINITE:
					(
						b = ReadByte filestream
						if (b==0) then
						(
							model.useFarAtten = true
						)else(
							model.useFarAtten = false
						)
					)
			 
				-- Overshoot
				GLS_LIGHT_OVERSHOOT:
					(
						b = ReadByte filestream
						if (model.type != #omni) do
						(
							if (b==0) then
							(
								model.overshoot = false
							)else(
								model.overshoot = true
							)
						)
					)	
			 
				-- Size
				GLS_LIGHT_RADIUS:
					b = ReadFloat filestream			
			 
				-- Color Red
				GLS_LIGHT_RED:
					r = ReadFloat filestream	
			 
				-- Color Blue
				GLS_LIGHT_GREEN:
					g = ReadFloat filestream	
			 
				-- Color Green
				GLS_LIGHT_BLUE:
					(
						b = ReadFloat filestream	
						r = r*255
						g = g*255
						b = b*255
						model.rgb = ([r,g,b] as color)
					)
				-- Color Multiplier
				GLS_LIGHT_INTENSITY:
					model.multiplier = ReadFloat filestream			
			 
				-- Range Near
				GLS_LIGHT_NEAR:
					model.farAttenStart = ReadFloat filestream
			 
				-- Range Far
				GLS_LIGHT_FAR:
					model.farAttenEnd = ReadFloat filestream			
					
				-- Cone Inner
				GLS_LIGHT_INNER:
					(
						if (model.type != #omni) then
						(
							model.hotspot = ReadFloat filestream
						)else(
							b = ReadFloat filestream
						)
					)
			 
				-- Cone Outer
				GLS_LIGHT_OUTER:
					(
						if (model.type != #omni) then
						(
							model.falloff = ReadFloat filestream
						)else(
							b = ReadFloat filestream
						)
					)
				
				-- Toon
				GLS_LIGHT_TOON:
					b = ReadByte filestream
	
				-- Toon Levels
				GLS_LIGHT_TOONLEVELS:
					b = ReadByte filestream
				
				-- Show Falloff
				GLS_LIGHT_SHOWFALLOFF:
					(
						b = ReadByte filestream
						if (b==0) then
						(
							model.showFarAtten = false
						)else(
							model.showFarAtten = true
						)
					)
				
				Default:
					gls_ExitChunk filestream
			)
		)
	
		return model
	)
	
	
	function gls_ReadNode filestream parent =(
	
		local motherchunk = gls_tos
		While (ftell filestream < gls_ChunkStack[motherchunk]) do
		(
			gls_ReadChunk filestream
			
			case gls_chunkid of
			(
	
				-- Node Name
				GLS_NODE_NAME:
					n = ReadString filestream
							
				-- Node Position
				GLS_NODE_POSITION:
					(
						px = ReadFloat filestream
						py = ReadFloat filestream
						pz = ReadFloat filestream
					)		
					
				-- Node Rotation	
				GLS_NODE_ROTATION:
					(
						rx = ReadFloat filestream
						ry = ReadFloat filestream
						rz = ReadFloat filestream
					)
							
				-- Node Scale
				GLS_NODE_SCALE:
					(
						sx = ReadFloat filestream
						sy = ReadFloat filestream
						sz = ReadFloat filestream
					)
								
				-- Node Custom Properties
				GLS_NODE_CUSTOMPROPS:
					c = ReadString filestream
				
				-- Hidden flag
				GLS_NODE_HIDDEN:
					hidden = ReadByte filestream
				
				-- A child node
				GLS_NODE:
					gls_ReadNode filestream model
				
				GLS_MESH:
					(
						model = gls_ReadMesh filestream
						model.parent	= parent
						model.name	 	= n
						if (parent!=undefined) do (
							model.pos		= parent.pos
							model.scale		= parent.scale
							model.rotation	= parent.rotation
						)
						in coordsys parent move model		[px,pz,py]
						in coordsys parent scale model  	[sx,sz,sy]
						in coordsys parent rotate model 	(eulerAngles -rx -rz ry)
					)
					
				GLS_LIGHT:
					(
						model = gls_ReadLight filestream
						model.parent	= parent
						model.name	 	= n
						if (parent!=undefined) do (
							model.pos		= parent.pos
							model.scale		= parent.scale
							model.rotation	= parent.rotation
						)
						in coordsys parent move model		[px,pz,py]
						in coordsys parent scale model  	[sx,sz,sy]
						-- This is most likely incorrect:
						in coordsys parent rotate model 	(eulerAngles rx ry -rz)
						
						-- The old variation...Might work better?
						-- in coordsys local rotate model 	(eulerAngles -rx -rz ry)
						-- in coordsys local rotate model 	(eulerAngles 0 0 90)
						
						-- If you can solve the rotation problem for lights
						-- please post it in the forum!
					)
					
				GLS_PIVOT:
					(
						model = gls_ReadPivot filestream
						model.parent	= parent
						model.name	 	= n
						if (parent!=undefined) do (
							model.pos		= parent.pos
							model.scale		= parent.scale
							model.rotation	= parent.rotation
						)
						in coordsys parent move model		[px,pz,py]
						in coordsys parent scale model  	[sx,sz,sy]
						in coordsys parent rotate model 	(eulerAngles -rx -rz ry)
					)
					
				Default:
					gls_ExitChunk filestream
			)
		)
	)
	
	function gls_ReadNodes filestream =(
	
		local motherchunk = gls_tos
		While (ftell filestream < gls_ChunkStack[motherchunk]) do
		(
			gls_ReadChunk filestream
			
			case gls_chunkid of
			(
				GLS_NODE:
					(
						gls_ReadNode filestream undefined
					)
			
				Default:
					gls_ExitChunk filestream
			)
		)
	)
	
	
	function gls_import = (
		
		import_name = getOpenFileName caption:"Import gile[s] scene" types:"gile[s] scene (*.gls)|*.gls|"
		
		if import_name != undefined then 
		(
			-- Get the file size
			tempf = openFile import_name
			if (tempf != undefined) then
			(
				seek tempf #eof
				gls_FileSize = filePos tempf
				close tempf
			)
	
			-- Clean up from last time
			ar_count = gls_texture.count
			for i = 1 to ar_count do
			(
				deleteItem gls_texture (ar_count+1-i)
			)
			ar_count = gls_alphatex.count
			for i = 1 to ar_count do
			(
				deleteItem gls_alphatex (ar_count+1-i)
			)
			ar_count = gls_lightmap.count
			for i = 1 to ar_count do
			(
				deleteItem gls_lightmap (ar_count+1-i)
			)
			ar_count = gls_material.count
			for i = 1 to ar_count do
			(
				deleteItem gls_material (ar_count+1-i)
			)
			gls_multimat = undefined
			
			-- Do the import
			f=fopen import_name "rb"
			if (f != undefined) then
			(
				try
				(
					gls_ReadChunk f
					if (gls_ChunkID == GLS_HEADER) then
					(
						s = ReadString f
						v = ReadFloat f
						-- format "File % opened, Size:%\n" import_name gls_FileSize
						-- format "This is a proper gile[s] scene, Data Check:%, %\n" s v
					
						progressstart "Importing gile[s] scene..."
					
						local motherchunk = gls_tos
						While (ftell f < gls_ChunkStack[motherchunk]) do
						(
							gls_ReadChunk f
					
							case gls_chunkid of
							(
								GLS_RENDER: 	gls_ExitChunk f
								GLS_AUTHOR:	 	gls_ExitChunk f
								GLS_EXPORTFILE: gls_ExitChunk f
								GLS_LIGHTMAPS:	gls_ReadLightmaps f
								GLS_TEXTURES:	gls_ReadTextures f
								GLS_MATERIALS:	gls_ReadMaterials f
								GLS_NODES:		gls_ReadNodes f
								Default:		gls_ExitChunk f
							)
						)
				
						progressEnd()
				
					)else(
						tx = "\""+import_name+"\"\n\nIs not a gile[s] scene!\n"
						messagebox tx title:"gile[s] scene import error!"
					)
				)
				catch
				(
					messagebox "Error importing .gls file!" title:"gile[s] scene import error!"
				)
				fclose f
			)
		)
	)
	
	on doimp pressed do
	(	
		gls_ChunkStack.count 	= 100
		gls_TOS = 1
		gls_matloader = rdo2.state
		
		if (rdo3.state == 1) do
		(
			max file new
		)
		disableSceneRedraw()
		gls_import()
		enableSceneRedraw()
		redrawViews()
	)
)
